package com.patrick.app;

import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

public class ConnectionLogAggregator {
    private List<ConnectionLog> connectionLogs;

    /**
     * ConnectionLogAggregator constructor.
     * @param connectionLogs List of collectionLogs from your Minecraft log.
     */
    public ConnectionLogAggregator(List<ConnectionLog> connectionLogs) {
        this.connectionLogs = connectionLogs;
    }

    
// The next two functions use some high level stuff. Basically you can boil it down to the following code:
// List<ConnectionLog> filteredLogs = List<ConnectionLog>();
// foreach(ConnectionLog log : this.connectionLogs) {
//     if (log.name == name) { //or xuid
//         filteredLogs.add(log);
//     }
// }
// return filteredLogs;

    /**
     * Gets your connection logs based on a XUID.
     * @param xuid Player's XUID.
     * @return A list of all the CollectionLogs for this XUID.
     */
    private List<ConnectionLog> getConnectionLogsByXuid(String xuid) {
        return this.connectionLogs.stream().filter(c -> c.getXuid().equals(xuid)).collect(Collectors.toList());
    }

    /**
     * Gets your connection logs based on a username.
     * @param name Player's username.
     * @return A list of all the CollectionLogs for this username.
     */
    private List<ConnectionLog> getConnectionLogsByName(String name) {
        return this.connectionLogs.stream().filter(c -> c.getName().equals(name)).collect(Collectors.toList());
    }

// The next couple functions calculate how long the user has played.
// The Duration class has some functions that can help us figure out the time between 2 times and add them together which we need to do both.
// I could've used some weird logic to get a connect/disconnect pair using their statuses but I decided to take a shortcut because since I know
// that unless someone is literally playing at the moment, there's always a pair of CONNECTS and DISCONNECTS.
// So I exploited that and used a counter that jumped by 2 to group them together.

// I actually originally had the code in getTimePlayedFromLogs in both functions.
// Once I realized they shared a lot of the same code, I was able to abstract that code into another function to reduce duplication.

    /**
     * From a list of ConnectionLogs, gets your duration played.
     * @param connectionLogs List of ColletionLogs
     * @return Duration object of total sum of time played.
     */
    private Duration getTimePlayedFromLogs(List<ConnectionLog> connectionLogs) {
        Duration sum = Duration.ZERO;
        int counter = 0;

        while(counter < connectionLogs.size() - 1) {
            ConnectionLog connected = connectionLogs.get(counter);
            ConnectionLog disconnected = connectionLogs.get(counter+1);

            System.out.println(connected.getTime().toString());
            System.out.println(disconnected.getTime().toString());
            System.out.println(Duration.between(connected.getTime(), disconnected.getTime()).toString());

            sum = sum.plus(Duration.between(connected.getTime(), disconnected.getTime()));
            counter += 2;
        }

        return sum;
    }

    /**
     * Calculates the time played by a XUID.
     * @param xuid Player's XUID.
     * @return Duration object of how long they've played.
     */
    public Duration getTimePlayedByXuid(String xuid) {
        List<ConnectionLog> connectionLogs = this.getConnectionLogsByXuid(xuid);
        return this.getTimePlayedFromLogs(connectionLogs);
    }

    /**
     * Calculates the time played by a username.
     * @param name Player's username.
     * @return Duration object of how long they've played.
     */
    public Duration getTimePlayedByName(String name) {
        List<ConnectionLog> connectionLogs = this.getConnectionLogsByName(name);
        return this.getTimePlayedFromLogs(connectionLogs);
    }
}

