package com.patrick.app;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.JDABuilder;
import net.dv8tion.jda.api.events.interaction.command.SlashCommandInteractionEvent;
import net.dv8tion.jda.api.hooks.ListenerAdapter;
import net.dv8tion.jda.api.interactions.commands.OptionType;
import net.dv8tion.jda.api.interactions.commands.build.Commands;
import net.dv8tion.jda.api.requests.GatewayIntent;

public class App extends ListenerAdapter {
    // args are the arguments that come after your command. We specifically for now only care about your Discord Bot Token.
    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("You have to provide a token as first argument!");
            System.exit(1);
        }

        // This creates the JDA object. This is copypasted.
        JDA jda = JDABuilder.createLight(args[0], Collections.<GatewayIntent>emptyList())
            .addEventListeners(new App())
            .build();

        // This helps set the commands that the bot will look for and adds Options.
        // In case you don't want to look it up:
        // .slash("String that is the slash command name", "String description that Discord shows to the user")
        // .addOption(OptionType of what input you're expecting, "String name of your option", "String description of the option", Boolean that says if it is required)
        jda.updateCommands().addCommands(
            Commands.slash("byxuid", "Get user amount time played by xuid")
                .addOption(OptionType.STRING, "xuid", "minecraft xuid", true),
            Commands.slash("byname", "Get user amount time played by ingame name (note: duplicate names will combine into one)")
                .addOption(OptionType.STRING, "name", "minecraft ingame name", true)
        ).queue();
    }
    
    @Override
    public void onSlashCommandInteraction(SlashCommandInteractionEvent event) {
        // Create default Duration object of ZERO time.
        Duration time = Duration.ZERO;

        // Createes a new ConnectionLogAggregator with the getConnectionLogs() list as our input
        ConnectionLogAggregator aggregator = new ConnectionLogAggregator(getConnectionLogs());

        System.out.println("received event: " + event.getName());

        // Based on the slash command event name, we can decide what to do
        // Either way, we basically just get whatever option the user gives (XUID/username) and call the corresponding getTimePlayed function.
        // default case is if it doesn't match either.
        switch (event.getName()) {
            case "byxuid":
                String xuid = event.getOptions().get(0).getAsString();
                System.out.println("getting time for: " + xuid);
                time = aggregator.getTimePlayedByXuid(xuid);
                break;
            case "byname":
                String name = event.getOptions().get(0).getAsString();
                System.out.println("getting time for: " + name);
                time = aggregator.getTimePlayedByName(name);
                break;
            default:
                System.out.printf("Unknown command %s used by %#s%n", event.getName(), event.getUser());
                event.reply("Error, please try again.");
                return;   
        }

        System.out.println("responding");

        // We have successfully gotten the time and should tell the user in a human-readable format.
        event.reply(String.format("%d hours, %d minutes, %d seconds", time.toHours(), time.toMinutesPart(), time.toSecondsPart())).queue();;
    }

    /**
     * Parses through "info.txt" where the Minecraft logs are supposed to be from.
     * @return List of ConnectionLogs
     */
    private List<ConnectionLog> getConnectionLogs() {
        // The variable we're going to store our ConnectionLogs in.
        List<ConnectionLog> connectionLogs = new ArrayList<ConnectionLog>();

        try {
            // Open file
            FileInputStream fstream = new FileInputStream("info.txt");
            BufferedReader br = new BufferedReader(new InputStreamReader(fstream));
            String strLine;

            // Read line by line
            while ((strLine = br.readLine()) != null)   {
                // Turns our string into a ConnectionLog object.
                ConnectionLog parsed = LogParser.parseLog(strLine);
                if (parsed != null) {
                    connectionLogs.add(parsed);
                }
            }
            System.out.println("done parsing");
            fstream.close();
        }
        catch (Exception e) {
            System.err.println("Error: " + e.getMessage());
        }

        return connectionLogs;

    }
}